//Class which adds a layer of abstraction and avoids having to worry about several issues:

// - "endian-ness":  When you create a BufferedFile for read or write, the first thing it does is read or
//    write a single byte flag to indicate the endianness of the file, and automatically converts to native
//    byte order if necessary when reading as long as you use the ReadNumber and WriteNumber functions.

// - size of numbers:  If I'm going down a structure with lots of members, writing each member, I find it
//   annoying to have to constantly check the type of each member, then call the appropriate sized byte
//   swap if necessary and write the appropriate size manually.  It is especially annoying when some members
//   are not base types (for example, time_t is not a base type - you have to know or look up to find out
//   that it is int64).  ReadNumber and WriteNumber accept any size or type of data (integers, floats, etc)
//   and treat it appropriately.  Thus, to write an array time_t TimeArray[5]
//      MyFile->WriteNumber(TimeArray,sizeof(time_t),5)

// - strings:  How do you write a string?  The length first, followed by the string, or omit the length
//   and search for the null terminator when you read?  BufferedFile provides ReadString() and WriteString()
//   to handle that for you.  No big deal but it simplifies things and keeps code more readable.

// - The last advantage is that it buffers the data internally during read and write, so that you can
//   conveniently read/write the data bits at a time (one member at a time, for example) without costly
//   repeated calls to the storage server.  Just using BFile, you should manually fill a write buffer for
//   the sake of efficiency and not call BFile::Read() or BFile::Write() too many times for small data
//   elements.  With buffering, it is just as efficient to write an int at a time.  This saves the hassle
//   by buffering output for you, allowing you to focus on the task at hand, namely writing your data one
//   piece at a time, and keeping your code simple and readable.  

//BufferedFile still provides all of the functionality of a BFile: It now provides Read/Write access, random
//access, etc.  All the regular BFile calls can be mixed with the BufferedFile calls, although the BFile calls
//won't benefit from the buffering (they force the buffer to flush, and actually introduce inefficiency - use
//them carefully; in general try to stick to the BufferedFile-specific calls)


#ifndef _SGB_BUFFERED_FILE_H_
#define _SGB_BUFFERED_FILE_H_


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <File.h>


//******************************************************************************************************
//**** BufferedFile CLASS DECLARATION
//******************************************************************************************************
class BufferedFile: public BFile
{
	public:
		//openMode may only be read or write, not read/write for all constructors
		BufferedFile(const entry_ref* ref, uint32 openMode, int32 BufferSize = 4095);
		BufferedFile(const BEntry* entry, uint32 openMode, int32 BufferSize = 4095);
		BufferedFile(const char* path, uint32 openMode, int32 BufferSize = 4095);
		BufferedFile(BDirectory* dir, const char* path, uint32 openMode, int32 BufferSize = 4095);

		virtual ~BufferedFile();	//Destroys the BufferedFile, but DOES NOT write any buffered data, as
									//a file error could occur without the opportunity to report the error.
									//You must call FlushBuffer() before destroying the BufferedFile in
									//order to write any data in the write buffer.
		status_t FlushBuffer();	//Flushes the read buffer and moves the file pointer to the position that
								//was being read from.  It is automatically called when necessary when
								//working within the file, but if you have buffered write data, call
								//it yourself before deleting the BufferedFile to make sure the data is 
								//written successfully (the destructor can't return an error code).
								//Returns B_NO_ERROR if the buffer was flushed successfully.
		status_t InitCheck();	//Returns B_NO_ERROR if the BufferedFile was created successfully.

		//Reads or writes the raw data contained in ReadBuffer or WriteBuffer.  Returns the number of bytes
		//actually read or written.
		int32 ReadRaw(void* Buffer, size_t BytesToRead);
		int32 WriteRaw(const void* Buffer, size_t BytesToWrite);

		//Reads or writes a string from the specified file.  Returns B_NO_ERROR if successful.  For
		//ReadString, the string will be null-terminated.  For WriteString, the string passed to it must be
		//null-terminated.  Failure could be due to reaching the end of the file before finishing the read
		//(returns B_FILE_ERROR), or due to running out of buffer space when reading (B_ERROR).
		status_t ReadString(char* Buffer, int32 BufferSize);
		status_t WriteString(const char* Buffer);

		//Read or write a number without regard to size, in native-endian format.  SizeOfData is the size
		//of EACH data element in bytes, not the whole buffer.  NumbersToRead or NumbersToWrite is the
		//number of data elements to read or write NOT the size of the whole buffer in bytes.  Example
		//usage:
		//   time_t TimeArray[5];
		//   MyFile->ReadNumber(TimeArray,sizeof(time_t),5)
		int32 ReadNumber(void* Buffer, uint8 SizeOfData, int32 NumbersToRead = 1);
		int32 WriteNumber(void* Buffer, uint8 SizeOfData, int32 NumbersToWrite = 1);

		bool IsBigEndian();		//Tells whether the file being read/written is BigEndian or not
		bool IsNativeEndian();	//Tells whether the file being read/written is already in native endian
								//format

		//BFile overrides
		virtual	ssize_t Read(void *buffer, size_t size);
		virtual	ssize_t ReadAt(off_t pos, void *buffer, size_t size);
		virtual	ssize_t Write(const void *buffer, size_t size);
		virtual	ssize_t WriteAt(off_t pos, const void *buffer, size_t size);
		virtual off_t Seek(off_t position, uint32 seek_mode);
		virtual	status_t SetSize(off_t size);

		//BFile functions not accessible from BFile as virtuals
		status_t SetTo(const entry_ref *ref, uint32 open_mode);
		status_t SetTo(const BEntry *entry, uint32 open_mode);
		status_t SetTo(const char *path, uint32 open_mode);
		status_t SetTo(const BDirectory *dir, const char *path, uint32 open_mode);

	private:
		void AllocAndInitBufferAndEndian(int32 Buffer_Size);
		void CheckNewFileStatus();

		int32 BufferSize;
		char* Buffer;
		int32 BytesInBuffer;
		int32 BufferPos;
		uint32 openMode;
		status_t Valid;
		bool BigEndian;
		bool HostBigEndian;
		bool WriteBuffered;
};


#endif